#include "DSP281x_Device.h"


// Private functions
Uint32 SCIA_GetLongData_app(void);
Uint16 SCIA_GetWordData_app(void);
Uint16 SCIA_GetByteData_app(void);
Uint16 SCIA_FIFO_GetWordData(void);
void SCIA_Init(void);
void SCIA_AutobaudLock(void);
void SCIB_AutobaudLock(void);
void SCIA_SendStatus(char *);
void SCIB_SendStatus(char *);
void	SCIA_SendHexWord(Uint16);
void	SCIA_SendHexWordReverse(Uint16);
void	SCIA_SendHexLong(Uint32);


//#################################################
// Uint32 SCIA_GetLongData(void)
//-----------------------------------------------
// This routine fetches two words from the SCI-A
// port and puts them together to form a single
// 32-bit value.  It is assumed that the host is
// sending the data in the form MSW:LSW.
//-----------------------------------------------
                                       
Uint32 SCIA_GetLongData_app()
{
    Uint32 longData = (Uint32)0x00000000;

    // Fetch the upper 1/2 of the 32-bit value 
    longData = ( (Uint32)SCIA_FIFO_GetWordData() << 16);
    
    // Fetch the lower 1/2 of the 32-bit value
    longData |= (Uint32)SCIA_FIFO_GetWordData();

    return longData;
}

Uint16 SCIA_GetByteData_app()
{
   Uint16 byteData;
  
   byteData = 0x0000;
   
   // Fetch byte and verify back to the host
   while(SciaRegs.SCIFFRX.bit.RXFIFST == 0) { } 
   byteData =  (Uint16)SciaRegs.SCIRXBUF.bit.RXDT;
   SciaRegs.SCITXBUF = byteData;
   
   return byteData;
}



//#################################################
// Uint16 SCIA_GetWordData(void)
//-----------------------------------------------
// This routine fetches two bytes from the SCI-A
// port and puts them together to form a single
// 16-bit value.  It is assumed that the host is
// sending the data in the order LSB followed by MSB.
//-----------------------------------------------


Uint16 SCIA_GetWordData_app()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(SciaRegs.SCIRXST.bit.RXRDY != 1) { } 
   wordData =  (Uint16)SciaRegs.SCIRXBUF.bit.RXDT;
   SciaRegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   while(SciaRegs.SCIRXST.bit.RXRDY != 1) { } 
   byteData =  (Uint16)SciaRegs.SCIRXBUF.bit.RXDT;
   SciaRegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}

Uint16 SCIA_FIFO_GetWordData()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(SciaRegs.SCIFFRX.bit.RXFIFST < 2) { } 

   wordData =  (Uint16)SciaRegs.SCIRXBUF.bit.RXDT;
//   SciaRegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   byteData =  (Uint16)SciaRegs.SCIRXBUF.bit.RXDT;
//   SciaRegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}


Uint16 SCIB_FIFO_GetWordData()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(ScibRegs.SCIFFRX.bit.RXFIFST < 2) { } 

   wordData =  (Uint16)ScibRegs.SCIRXBUF.bit.RXDT;
//   SciaRegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   byteData =  (Uint16)ScibRegs.SCIRXBUF.bit.RXDT;
//   SciaRegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}




//#################################################
// void SCIA_Init(void)
//-----------------------------------------------------
// This routine initializes the SCI
// 
//-----------------------------------------------------


void SCIA_Init()
{

    SciaRegs.SCIHBAUD = 0;
    SciaRegs.SCILBAUD = 19;

    // 1 stop bit, No parity, 8-bit character
    // No loopback
    SciaRegs.SCICCR.all = 0x0007;
    
    // Enable TX, RX, Use internal SCICLK
    SciaRegs.SCICTL1.all = 0x0003; 

    // Disable RxErr, Sleep, TX Wake, 
    // Diable Rx Interrupt, Tx Interrupt
    SciaRegs.SCICTL2.all = 0x0000;

    // Relinquish SCI-A from reset
    SciaRegs.SCICTL1.all = 0x0023;
     
    return;
}

void SCIB_Init()
{

    ScibRegs.SCIHBAUD = 0;
    ScibRegs.SCILBAUD = 118;

    // 1 stop bit, No parity, 8-bit character
    // No loopback
    ScibRegs.SCICCR.all = 0x0007;
    
    // Enable TX, RX, Use internal SCICLK
    ScibRegs.SCICTL1.all = 0x0003; 

    // Disable RxErr, Sleep, TX Wake, 
    // Diable Rx Interrupt, Tx Interrupt
    ScibRegs.SCICTL2.all = 0x0000;

    // Relinquish SCI-A from reset
    ScibRegs.SCICTL1.all = 0x0023;
     
    return;
}

//#################################################
// void SCIA_AutobaudLock(void)
//-----------------------------------------------------
// This routine is updates the SCIA baud rate
// using the auto-baud logic
// 
//-----------------------------------------------------



void SCIA_AutobaudLock()
{
    Uint16 byteData;

    // Must prime baud register with >= 1
    SciaRegs.SCIHBAUD = 0;
    SciaRegs.SCILBAUD = 1;
    
    // Prepare for autobaud detection
    // Set the CDC bit to enable autobaud detection
    // and clear the ABD bit      
	SciaRegs.SCIFFCT.bit.ABDCLR = 1;
    SciaRegs.SCIFFCT.all = 0x2000;
    // Wait until we correctly read an 
    // 'A' or 'a' and lock    
    while(SciaRegs.SCIFFCT.bit.ABD != 1) {}

   // After autobaud lock, clear the CDC bit
    SciaRegs.SCIFFCT.bit.CDC = 0;

    while(SciaRegs.SCIRXST.bit.RXRDY != 1) { } 
    byteData = SciaRegs.SCIRXBUF.bit.RXDT;
    SciaRegs.SCITXBUF = byteData;

    return;   
}

void SCIB_AutobaudLock()
{
    Uint16 byteData;

    // Must prime baud register with >= 1
    ScibRegs.SCIHBAUD = 0;
    ScibRegs.SCILBAUD = 1;
    
    // Prepare for autobaud detection
    // Set the CDC bit to enable autobaud detection
    // and clear the ABD bit      
	ScibRegs.SCIFFCT.bit.ABDCLR = 1;
    ScibRegs.SCIFFCT.all = 0x2000;
    // Wait until we correctly read an 
    // 'A' or 'a' and lock    
    while(ScibRegs.SCIFFCT.bit.ABD != 1) {}

   // After autobaud lock, clear the CDC bit
    ScibRegs.SCIFFCT.bit.CDC = 0;

    while(ScibRegs.SCIRXST.bit.RXRDY != 1) { } 
    byteData = ScibRegs.SCIRXBUF.bit.RXDT;
    ScibRegs.SCITXBUF = byteData;

    return;   
}



void	SCIA_SendHexWord(Uint16 temp1)
{
	while(SciaRegs.SCIFFTX.bit.TXFFST>14) ;
	
	SciaRegs.SCITXBUF = temp1>>8;
	SciaRegs.SCITXBUF = temp1;

	return;
}

void	SCIA_SendHexWordReverse(Uint16 temp1)
{
	while(SciaRegs.SCIFFTX.bit.TXFFST>14) ;
	
	SciaRegs.SCITXBUF = temp1;
	SciaRegs.SCITXBUF = temp1>>8;

	return;
}


void	SCIA_SendHexLong(Uint32 temp1)
{
	while(SciaRegs.SCIFFTX.bit.TXFFST>12) ;
	
	SciaRegs.SCITXBUF = temp1>>24;
	SciaRegs.SCITXBUF = temp1>>16;
	SciaRegs.SCITXBUF = temp1>>8;
	SciaRegs.SCITXBUF = temp1;

	return;
}


void SCIA_SendStatus(char *msg)
{
	while(*msg)
	{
		while((SciaRegs.SCIFFTX.bit.TXFFST<16) && *msg) 
			SciaRegs.SCITXBUF = *msg++;

		while(SciaRegs.SCIFFTX.bit.TXFFST) ;	//TX FIFO empty?
	}

    return;   

}

void SCIB_SendStatus(char *msg)
{
	while(*msg)
	{
		while((ScibRegs.SCIFFTX.bit.TXFFST<16) && *msg) 
			ScibRegs.SCITXBUF = *msg++;

		while(ScibRegs.SCIFFTX.bit.TXFFST) ;	//TX FIFO empty?
	}

    return;   

}





// EOF-------

